﻿################################################################################################
# This script is not supported under any Microsoft standard support program or service.
# This script is provided AS IS without warranty of any kind.
# Microsoft further disclaims all implied warranties including, without limitation, any implied
# warranties of merchantability or of fitness for a particular purpose. The entire risk arising
# out of the use or performance of the sample script and documentation remains with you. In no
# event shall Microsoft, its authors, or anyone else involved in the creation, production, or
# delivery of the scripts be liable for any damages whatsoever (including, without limitation,
# damages for loss of business profits, business interruption, loss of business information,
# or other pecuniary loss) arising out of the use of or inability to use the sample script or
# documentation, even if Microsoft has been advised of the possibility of such damages.
################################################################################################
<#
	.SYNOPSIS
	This script sets up pre-requisites for running DSC in order to cover STIG gaps.

	Additionally, this policy sets the Data Execution Prevention policy to OptOut.
#>

[CmdletBinding()]
param ()

$scriptVer = '3.1'

# Ensure that the proper DSC modules are installed
$destination = "$env:ProgramFiles\WindowsPowerShell\Modules"
$secmodule = Get-Item -Path "$PSScriptRoot\Modules\SecurityPolicyDsc"
$secpresent = Get-InstalledModule | Where-Object -FilterScript {$_.Name -eq 'SecurityPolicyDsc'}
$psDscModule = Get-Item -Path "$PSScriptRoot\Modules\PSDscResources"
$psDscPresent = Get-InstalledModule | Where-Object -FilterScript {$_.Name -eq 'PSDscResources'}

if ($null -eq $secpresent -or $secpresent.Version -lt '2.10.0.0') {
    try {
        Copy-Item -Path $secmodule -Destination $destination -Recurse -Force
    }
    catch {
        Write-Warning $Error[0]
    }
}
else {
    Write-Host "SecurityPolicyDsc module $($secpresent.Version.ToString()) is already installed."
}

if ($null -eq $psDscPresent -or $psDscPresent.Version -lt '2.12.0.0') {
    try {
        Copy-Item -Path $psDscModule -Destination $destination -Recurse -Force
    }
    catch {
        Write-Warning $Error[0]
    }
}
else {
    Write-Host "PS DSC Resources module $($psDscPresent.Version.ToString()) is already installed."
}

# Set up WinRM for DSC
Enable-PSRemoting -SkipNetworkProfileCheck

# Disable Optional Feature Windows PowerShell V2 Root
Disable-WindowsOptionalFeature -Online -FeatureName MicrosoftWindowsPowerShellV2Root

# Determines bitlocker encrypted volumes
$encryptedVolumes = (Get-BitLockerVolume | Where-Object {$_.ProtectionStatus -eq 'On'}).MountPoint

<# Enables DEP. If there are bitlocker encrypted volumes, bitlocker is temporarily suspended for this operation
Data Execution Prevention (DEP) must be configured to at least OptOut
V-220726 Windows 10
V-253283 Windows 11
#>
if ($encryptedVolumes.Count -gt 0) {
    foreach ($volume in $encryptedVolumes) {
        Suspend-BitLocker -MountPoint $volume
    }

    Start-Process -Wait -FilePath 'C:\Windows\System32\bcdedit.exe' -ArgumentList '/set "{current}" nx OptOut'

    foreach ($volume in $encryptedVolumes) {
        Resume-BitLocker -MountPoint $volume
    }
}
else {
    Start-Process -Wait -FilePath 'C:\Windows\System32\bcdedit.exe' -ArgumentList '/set "{current}" nx OptOut'
}

# Copy user registry settings script to local machine
Copy-Item $PSScriptRoot\Manage-UserSettings.ps1 -Destination "$env:SystemRoot\SysWOW64\DISASTIGs" -Force
Copy-Item $PSScriptRoot\User-SettingsData.psd1 -Destination "$env:SystemRoot\SysWOW64\DISASTIGs" -Force

# Set up scheduled task to ensure user registry settings are present on login
$argument = '-WindowStyle Hidden -ExecutionPolicy Bypass -Command "&{%systemroot%\SysWOW64\DISASTIGs\Manage-UserSettings.ps1}"'

# Create logon scheduled task to manage user registry settings
$action = New-ScheduledTaskAction -Execute 'PowerShell.exe' -Argument $argument
$principal = New-ScheduledTaskPrincipal -UserID 'NT Authority\System'
$settings = New-ScheduledTaskSettingsSet -AllowStartIfOnBatteries -DontStopIfGoingOnBatteries -DontStopOnIdleEnd -ExecutionTimeLimit (New-TimeSpan -Minutes 20) -Compatibility Win8
$trigger = New-ScheduledTaskTrigger -AtLogOn
$task = New-ScheduledTask -Action $action -Principal $principal -Settings $settings -Trigger $trigger
$scheduledTask = Get-ScheduledTask -TaskName 'Manage User STIG Settings' -ErrorAction SilentlyContinue

if ($null -eq $scheduledTask) {
    try {
        Register-ScheduledTask -TaskName 'Manage User STIG Settings' -InputObject $task
    }
    catch {
        Write-Error 'Error creating task to manage user STIG settings on login'
    }
}
else {
    try {
        Unregister-ScheduledTask -TaskName 'Manage User STIG Settings' -Confirm:$false
        Register-ScheduledTask -TaskName 'Manage User STIG Settings' -InputObject $task
    }
    catch {
        Write-Error 'Error creating task to manage user STIG settings on login'
    }
}
